﻿using iText.Html2pdf;
using iText.Html2pdf.Attach.Impl;
using iText.IO.Font;
using iText.Kernel.Colors;
using iText.Kernel.Font;
using iText.Kernel.Geom;
using iText.Kernel.Pdf;
using iText.Kernel.Pdf.Canvas;
using iText.Layout.Font;
using iText.Layout.Renderer;
using NVCC.WebUI.Infrastructure.iTextCustom;
using System;
using System.IO;
using System.Web;

namespace NVCC.WebUI.Infrastructure
{
    public class AccessiblePdfService : IAccessiblePdfService
    {
        
        public string createPdf(string html, string leftFooter, PdfTypes type = PdfTypes.RefDoc, bool includeCenterFooter = true)
        {
            var pdfWithoutFootersPath = System.IO.Path.GetTempFileName();

            FileStream outputStream = new FileStream(pdfWithoutFootersPath, FileMode.Create, FileAccess.ReadWrite);

            WriterProperties writerProperties = new WriterProperties();

            //Add metadata
            writerProperties.AddUAXmpMetadata();

            PdfWriter pdfWriter = new PdfWriter(outputStream, writerProperties);
            

            PdfDocument pdfDoc = new PdfDocument(pdfWriter);
            pdfDoc.GetCatalog().SetLang(new PdfString("en-US"));

            //Set the document to be tagged
            pdfDoc.SetTagged();
            pdfDoc.GetCatalog().SetViewerPreferences(new PdfViewerPreferences().SetDisplayDocTitle(true));

            //Set meta tags
            PdfDocumentInfo pdfMetaData = pdfDoc.GetDocumentInfo();
            pdfMetaData.SetAuthor("US Department of Veterans Affairs");
            pdfMetaData.AddCreationDate();
            pdfMetaData.GetProducer();
            pdfMetaData.SetCreator("VA Community Care Referral Documentation Tool");

            pdfDoc.SetDefaultPageSize(PageSize.LETTER);

            switch (type)
            {
                case PdfTypes.RefDoc:
                    pdfMetaData.SetKeywords("patient referral, medical history");
                    pdfMetaData.SetSubject("Referral for Care");
                    break;
                case PdfTypes.PatientRoi:
                    pdfMetaData.SetKeywords("Accounting of Disclosure report, ROI, AOD, Release of information");
                    pdfMetaData.SetSubject("VHA Accounting of Disclosure Search");
                    break;
                case PdfTypes.RoiAdmin:
                    pdfMetaData.SetKeywords("ROI, ROI Admin, report");
                    pdfMetaData.SetSubject("ROI Admin Report");
                    break;
                default:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new ArgumentOutOfRangeException("type", "Not a valid RefDoc PDF type"));
                    break;
            }
            
            ConverterProperties props = new ConverterProperties();

            FontProvider fp = new FontProvider();
            fp.AddDirectory(HttpContext.Current.Server.MapPath("/Content/Fonts/"));
            props.SetFontProvider(fp);

            DefaultTagWorkerFactory tagWorkerFactory = new AccessibilityTagWorkerFactory();

            props.SetTagWorkerFactory(tagWorkerFactory);

            props.SetOutlineHandler(OutlineHandler.CreateStandardHandler());

            HtmlConverter.ConvertToPdf(html, pdfDoc, props);

            pdfDoc.Close();
            
            var pdfWithFootersPath = InsertFooters(pdfWithoutFootersPath, leftFooter, includeCenterFooter);

            try
            {
                File.Delete(pdfWithoutFootersPath);
            }
            catch (Exception ex)
            {
                Elmah.ErrorSignal.FromCurrentContext().Raise(ex);
            }

            return pdfWithFootersPath;

        }


        private string InsertFooters(string filePath, string leftFooter, bool includeCenterFooter)
        {
            var pdfWithFootersPath = System.IO.Path.GetTempFileName();
            PdfDocument pdfDoc = new PdfDocument(new PdfReader(filePath), new PdfWriter(pdfWithFootersPath));
            Rectangle pageSize;
            PdfCanvas canvas;
            var arialPath = HttpContext.Current.Server.MapPath("/Content/Fonts/ARIAL.TTF");
            var arialFont = PdfFontFactory.CreateFont(arialPath, true);
            //I am fairly certain this line is not needed because the Arial font is already added.
            //Also, I'm not sure if there are any negative consequences of adding it twice so I'm commenting it out for now:
            //pdfDoc.AddFont(arialFont);
            int n = pdfDoc.GetNumberOfPages();
            for (int i = 1; i <= n; i++)
            {
                PdfPage page = pdfDoc.GetPage(i);
                pageSize = page.GetPageSize();
                canvas = new PdfCanvas(page);
                Rectangle box = new Rectangle(10, 5, pageSize.GetWidth() - 20, 25);

                canvas.OpenTag(new CanvasArtifact());

                //Draw footer line
                canvas.SetStrokeColorGray(0.2f)
                        .SetLineWidth(.2f)
                        .MoveTo(30, 20)
                        .LineTo(pageSize.GetWidth() - 30, 20).Stroke();

                //Draw page number
                canvas.BeginText().SetFontAndSize(
                        arialFont, 7)
                        .MoveText(pageSize.GetWidth() - 67, 10)
                        .ShowText("Page ")
                        .ShowText(i.ToString())
                        .ShowText(" of ")
                        .ShowText(n.ToString())
                        .EndText();

                if (includeCenterFooter)
                {
                    canvas.BeginText().SetFontAndSize(
                            arialFont, 7)
                            .MoveText(pageSize.GetWidth() / 2 - 58, 10)
                            .ShowText("Document Created: ")
                            .ShowText(DateTime.Now.ToString("MM/dd/yyyy HH:mm"))
                            .EndText();
                }

                canvas.BeginText().SetFontAndSize(
                        arialFont, 7)
                        .MoveText(30, 10)
                        .ShowText(leftFooter)
                        .EndText();

                canvas.CloseTag();
            }
            pdfDoc.Close();
            return pdfWithFootersPath;
        }
    }
}